"""
Causal forest tuning

Parameters
----------
  X : The covariates used in the regression

  y : The outcome

  w : The treatment assignment (may be binary or real)

  Y.hat : Estimates of the expected responses E[Y | Xi], marginalizing
              over treatment. See section 6.1.1 of the GRF paper for
              further discussion of this quantity

  W.hat : Estimates of the treatment propensities E[W | Xi]
  
  sample.weights : (experimental) Weights given to an observation in estimation.
                        If NULL, each observation is given the same weight. Default is NULL

  clusters : Vector of integers or factors specifying which cluster each observation corresponds to.
                  Default is NULL (ignored)

  samples.per.cluster : If sampling by cluster, the number of observations to be sampled from
                        each cluster when training a tree. If NULL, we set samples.per.cluster to the size
                        of the smallest cluster. If some clusters are smaller than samples.per.cluster,
                        the whole cluster is used every time the cluster is drawn. Note that
                        clusters with less than samples.per.cluster observations get relatively
                        smaller weight than others in training the forest, i.e., the contribution
                        of a given cluster to the final forest scales with the minimum of
                        the number of observations in the cluster and samples.per.cluster. Default is NULL.

  sample.fraction : Fraction of the data used to build each tree.
                        Note: If honesty = True, these subsamples will
                        further be cut by a factor of honesty.fraction. Default is 0.5

  mtry : Number of variables tried for each split. Default is
              \eqn{\sqrt p + 20} where p is the number of variables

  min.node.size : A target for the minimum number of observations in each tree leaf. Note that nodes
                  with size smaller than min.node.size can occur, as in the original randomForest package.
                  Default is 5

  honesty : Whether to use honest splitting (i.e., sub-sample splitting). Default is True.
            For a detailed description of honesty, honesty.fraction, honesty.prune.leaves, and recommendations for
            parameter tuning, see the grf
            \href{https://grf-labs.github.io/grf/REFERENCE.html#honesty-honesty-fraction-honesty-prune-leaves}{algorithm reference}.
  
  honesty.fraction : The fraction of data that will be used for determining splits if honesty = True. Corresponds
                     to set J1 in the notation of the paper. Default is 0.5 (i.e. half of the data is used for
                     determining splits)

  honesty.prune.leaves : If True, prunes the estimation sample tree such that no leaves
                         are empty. If FALSE, keep the same tree as determined in the splits sample (if an empty leave is encountered, that
                         tree is skipped and does not contribute to the estimate). Setting this to FALSE may improve performance on
                         small/marginally powered data, but requires more trees (note: tuning does not adjust the number of trees).
                         Only applies if honesty is enabled. Default is True.
  
  alpha : A tuning parameter that controls the maximum imbalance of a split. Default is 0.05
  
  imbalance.penalty : A tuning parameter that controls how harshly imbalanced splits are penalized. Default is 0.
  
  stabilize.splits : Whether or not the treatment should be taken into account when
                     determining the imbalance of a split. Default is True

  ci.group.size : The forest will grow ci.group.size trees on each subsample.
                  In order to provide confidence intervals, ci.group.size must
                  be at least 2. Default is 2.
                  
  tune.parameters : A vector of parameter names to tune.
                    If "all": all tunable parameters are tuned by cross-validation. The following parameters are
                    tunable: ("sample.fraction", "mtry", "min.node.size", "honesty.fraction",
                    "honesty.prune.leaves", "alpha", "imbalance.penalty"). If honesty is FALSE the honesty.* parameters are not tuned.
                    Default is "all"

  tune.num.trees : The number of trees in each 'mini forest' used to fit the tuning model. Default is 50

  tune.num.reps : The number of forests used to fit the tuning model. Default is 100

  tune.num.draws : The number of random parameter values considered when using the model
                   to select the optimal parameters. Default is 1000

  num.threads : Number of threads used in training. By default, the number of threads is set
                to the maximum hardware concurrency

  seed : The seed of the C++ random number generator.

Returns
-------
  A list consisting of the optimal parameter values ('params') along with their debiased
  error ('error').

@examples
\dontrun{
# Find the optimal tuning parameters.
n = 500
p = 10
X = matrix(rnorm(n * p), n, p)
W = rbinom(n, 1, 0.5)
Y = pmax(X[, 1], 0) * W + X[, 2] + pmin(X[, 3], 0) + rnorm(n)
Y.hat = predict(regression_forest(X, Y))$predictions
W.hat = rep(0.5, n)
params = tune_causal_forest(X, Y, W, Y.hat, W.hat)$params

# Use these parameters to train a regression forest.
tuned.forest = causal_forest(X, Y, W,
  Y.hat = Y.hat, W.hat = W.hat, num.trees = 1000,
  min.node.size = as.numeric(params["min.node.size"]),
  sample.fraction = as.numeric(params["sample.fraction"]),
  mtry = as.numeric(params["mtry"]),
  alpha = as.numeric(params["alpha"]),
  imbalance.penalty = as.numeric(params["imbalance.penalty"])
)
}

@export
"""
tune_causal_forest = function(X, Y, W, Y.hat, W.hat,
                              sample.weights = NULL,
                              clusters = NULL,
                              samples.per.cluster = NULL,
                              sample.fraction = 0.5,
                              mtry = min(ceiling(sqrt(ncol(X)) + 20), ncol(X)),
                              min.node.size = 5,
                              honesty = True,
                              honesty.fraction = 0.5,
                              honesty.prune.leaves = True,
                              alpha = 0.05,
                              imbalance.penalty = 0,
                              stabilize.splits = True,
                              ci.group.size = 2,
                              tune.parameters = "all",
                              tune.num.trees = 200,
                              tune.num.reps = 50,
                              tune.num.draws = 1000,
                              num.threads = NULL,
                              seed = runif(1, 0, .Machine$integer.max)) {
  validate_X(X)
  validate_sample_weights(sample.weights, X)
  Y = validate_observations(Y, X)
  W = validate_observations(W, X)
  clusters = validate_clusters(clusters, X)
  samples.per.cluster = validate_samples_per_cluster(samples.per.cluster, clusters)
  num.threads = validate_num_threads(num.threads)

  all.tunable.params = c("sample.fraction", "mtry", "min.node.size", "honesty.fraction",
                          "honesty.prune.leaves", "alpha", "imbalance.penalty")

  default.parameters = list(sample.fraction = 0.5,
                             mtry = min(ceiling(sqrt(ncol(X)) + 20), ncol(X)),
                             min.node.size = 5,
                             honesty.fraction = 0.5,
                             honesty.prune.leaves = True,
                             alpha = 0.05,
                             imbalance.penalty = 0)

  Y.centered = Y - Y.hat
  W.centered = W - W.hat
  data = create_data_matrices(X, outcome = Y.centered, treatment = W.centered,
                              sample.weights = sample.weights)
  nrow.X = nrow(X)
  ncol.X = ncol(X)
  args = list(clusters = clusters,
               samples.per.cluster = samples.per.cluster,
               sample.fraction = sample.fraction,
               mtry = mtry,
               min.node.size = min.node.size,
               honesty = honesty,
               honesty.fraction = honesty.fraction,
               honesty.prune.leaves = honesty.prune.leaves,
               alpha = alpha,
               stabilize.splits = stabilize.splits,
               imbalance.penalty = imbalance.penalty,
               ci.group.size = ci.group.size,
               num.threads = num.threads,
               seed = seed,
               reduced.form.weight = 0)

  if (identical(tune.parameters, "all")) {
    tune.parameters = all.tunable.params
  } else {
    tune.parameters = unique(match.arg(tune.parameters, all.tunable.params, several.ok = True))
  }
  if (!honesty) {
    tune.parameters = tune.parameters[!grepl("honesty", tune.parameters)]
  }

  tune.parameters.defaults = default.parameters[tune.parameters]
  train = causal_train
  predict_oob = causal_predict_oob
  predict.data.args = c("train.matrix", "sparse.train.matrix", "outcome.index", "treatment.index")

  tuning.output = tune_forest(data = data,
                               nrow.X = nrow.X,
                               ncol.X = ncol.X,
                               args = args,
                               tune.parameters = tune.parameters,
                               tune.parameters.defaults = tune.parameters.defaults,
                               num.fit.trees = tune.num.trees,
                               num.fit.reps = tune.num.reps,
                               num.optimize.reps = tune.num.draws,
                               train = train,
                               predict_oob = predict_oob,
                               predict.data.args = predict.data.args)

  tuning.output
}